<?php
/**
 * Plugin Name: Phajay Payment Integration (With Documentation)
 * Description: ระบบชำระเงิน Phajay (Link / QR / Credit Card) พร้อมคู่มือการใช้งานในตัว
 * Version: 1.9
 * Author: Your Name
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; 
}

class Phajay_Payment_Plugin {

    // API Endpoints
    private $api_link_url   = 'https://payment-gateway.phajay.co/v1/api/link/payment-link';
    private $api_credit_url = 'https://payment-gateway.phajay.co/v1/api/jdb2c2p/payment/payment-link';
    
    private $api_qr_endpoints = array(
        'bcel' => 'https://payment-gateway.phajay.co/v1/api/payment/generate-bcel-qr',
        'jdb'  => 'https://payment-gateway.phajay.co/v1/api/payment/generate-jdb-qr',
        'ldb'  => 'https://payment-gateway.phajay.co/v1/api/payment/generate-ldb-qr',
        'ib'   => 'https://payment-gateway.phajay.co/v1/api/payment/generate-ib-qr'
    );

    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_shortcode('phajay_pay_button', array($this, 'render_shortcode'));
        add_action('admin_post_phajay_process_payment', array($this, 'process_payment'));
        add_action('admin_post_nopriv_phajay_process_payment', array($this, 'process_payment'));
        add_action('wp_head', array($this, 'print_custom_styles'));
    }

    // --- 1. Admin Settings ---
    public function add_admin_menu() {
        add_options_page('Phajay Settings', 'Phajay Payment', 'manage_options', 'phajay-payment', array($this, 'settings_page_html'));
    }

    public function register_settings() {
        register_setting('phajay_group', 'phajay_secret_key');
        
        // Toggle Options
        register_setting('phajay_group', 'phajay_enable_link');
        register_setting('phajay_group', 'phajay_enable_qr');
        register_setting('phajay_group', 'phajay_enable_credit');
        
        // Styles
        register_setting('phajay_group', 'phajay_btn_label');
        register_setting('phajay_group', 'phajay_btn_bg_color');
        register_setting('phajay_group', 'phajay_btn_text_color');
        register_setting('phajay_group', 'phajay_btn_radius');
    }

    public function settings_page_html() {
        ?>
        <div class="wrap">
            <h1>ตั้งค่า Phajay Payment Integration</h1>
            
            <div style="display:flex; gap:20px; flex-wrap:wrap;">
                <div style="flex: 1; min-width: 300px; background:#fff; padding:20px; border:1px solid #ccd0d4; box-shadow:0 1px 1px rgba(0,0,0,.04);">
                    <form method="post" action="options.php">
                        <?php settings_fields('phajay_group'); ?>
                        <h2 style="margin-top:0;">1. การเชื่อมต่อ & ตัวเลือก</h2>
                        <p>นำ Secret Key จาก <a href="https://portal.phajay.co/" target="_blank">Phajay Portal</a> มาใส่ที่นี่</p>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">Secret Key</th>
                                <td><input type="text" name="phajay_secret_key" value="<?php echo esc_attr(get_option('phajay_secret_key')); ?>" class="regular-text" /></td>
                            </tr>
                            
                            <tr><th colspan="2"><hr></th></tr>
                            <tr><th colspan="2"><h3>เปิด/ปิด ช่องทางชำระเงิน</h3></th></tr>
                            
                            <tr>
                                <th scope="row">Payment Link</th>
                                <td><label><input type="checkbox" name="phajay_enable_link" value="1" <?php checked(get_option('phajay_enable_link', '1'), 1); ?>> เปิดใช้งาน (Redirect ไปหน้าเว็บรวม)</label></td>
                            </tr>
                            <tr>
                                <th scope="row">Scan QR</th>
                                <td><label><input type="checkbox" name="phajay_enable_qr" value="1" <?php checked(get_option('phajay_enable_qr', '1'), 1); ?>> เปิดใช้งาน (แสดง QR Code ทันที)</label></td>
                            </tr>
                            <tr>
                                <th scope="row">Credit Card</th>
                                <td><label><input type="checkbox" name="phajay_enable_credit" value="1" <?php checked(get_option('phajay_enable_credit', '1'), 1); ?>> เปิดใช้งาน (ตัดบัตรเครดิต)</label></td>
                            </tr>

                            <tr><th colspan="2"><hr></th></tr>
                            <tr><th colspan="2"><h3>ดีไซน์ปุ่ม</h3></th></tr>
                            <tr>
                                <th scope="row">ข้อความบนปุ่ม</th>
                                <td><input type="text" name="phajay_btn_label" value="<?php echo esc_attr(get_option('phajay_btn_label', 'ดำเนินการชำระเงิน')); ?>" class="regular-text" /></td>
                            </tr>
                            <tr>
                                <th scope="row">สีพื้นหลัง</th>
                                <td><input type="color" name="phajay_btn_bg_color" value="<?php echo esc_attr(get_option('phajay_btn_bg_color', '#2271b1')); ?>" /></td>
                            </tr>
                            <tr>
                                <th scope="row">สีตัวอักษร</th>
                                <td><input type="color" name="phajay_btn_text_color" value="<?php echo esc_attr(get_option('phajay_btn_text_color', '#ffffff')); ?>" /></td>
                            </tr>
                            <tr>
                                <th scope="row">Border Radius</th>
                                <td><input type="number" name="phajay_btn_radius" value="<?php echo esc_attr(get_option('phajay_btn_radius', '5')); ?>" class="small-text" /> px</td>
                            </tr>
                        </table>
                        <?php submit_button(); ?>
                    </form>
                </div>

                <div style="flex: 1; min-width: 300px; background:#fff; padding:20px; border:1px solid #ccd0d4; box-shadow:0 1px 1px rgba(0,0,0,.04);">
                    <h2 style="margin-top:0;">📘 คู่มือการใช้งาน (How to Use)</h2>
                    <hr>
                    
                    <h3>1. การใช้งานพื้นฐาน (Basic Usage)</h3>
                    <p>นำ Shortcode นี้ไปวางในหน้า Page หรือ Post ที่ต้องการแสดงปุ่ม:</p>
                    <code style="display:block; background:#f0f0f1; padding:10px; margin-bottom:10px;">[phajay_pay_button amount="100000" description="ค่าสินค้า"]</code>
                    <ul>
                        <li><strong>amount:</strong> ยอดเงินที่ต้องการเรียกเก็บ (LAK)</li>
                        <li><strong>description:</strong> รายละเอียดสินค้า</li>
                        <li><strong>label:</strong> ข้อความบนปุ่ม (ถ้าไม่ใส่ จะใช้ค่า Default)</li>
                    </ul>

                    <hr>

                    <h3>2. การรับค่ายอดเงินแบบอัตโนมัติ (Dynamic)</h3>
                    <p>หากต้องการส่งลิงก์ให้ลูกค้าพร้อมระบุยอดเงิน ให้สร้างหน้าเว็บที่มี Shortcode <code>[phajay_pay_button]</code> แล้วส่งลิงก์แบบนี้:</p>
                    <code style="display:block; background:#f0f0f1; padding:10px; margin-bottom:10px;">https://yourwebsite.com/pay?pay_amount=250000</code>
                    <p>ระบบจะดึงยอด <strong>250,000</strong> มาแสดงให้อัตโนมัติ</p>

                    <hr>

                    <h3>3. สำหรับนักพัฒนา (Developer PHP)</h3>
                    <p>หากต้องการใช้ในไฟล์ธีม (เช่น <code>single-product.php</code>):</p>
                    <pre style="background:#f0f0f1; padding:10px; overflow-x:auto;"><?php echo htmlspecialchars("<?php echo do_shortcode('[phajay_pay_button amount=\"' . \$price . '\" description=\"' . \$product_name . '\"]'); ?>"); ?></pre>

                    <hr>

                    <h3>❓ FAQ คำถามที่พบบ่อย</h3>
                    <p><strong>Q: จ่ายเสร็จแล้วให้เด้งไปหน้าไหน?</strong><br>
                    A: ไปตั้งค่า <em>Success URL</em> ที่เว็บ <a href="https://portal.phajay.co/" target="_blank">Phajay Portal</a> (เมนู Settings > Callback URL)</p>
                    
                    <p><strong>Q: QR Code ไม่ขึ้น?</strong><br>
                    A: ตรวจสอบ <em>Secret Key</em> ว่าถูกต้องหรือไม่ และเซิร์ฟเวอร์ต้องรองรับการเชื่อมต่อภายนอก</p>
                </div>
            </div>
        </div>
        <?php
    }

    public function print_custom_styles() {
        $bg = get_option('phajay_btn_bg_color', '#2271b1');
        $text = get_option('phajay_btn_text_color', '#ffffff');
        $radius = get_option('phajay_btn_radius', '5');
        ?>
        <style>
            .phajay-wrapper { 
                background: #f9f9f9; padding: 20px; border-radius: 8px; border: 1px solid #e5e5e5; max-width: 400px;
            }
            .phajay-option { margin-bottom: 12px; display: flex; align-items: center; cursor: pointer; }
            .phajay-option input { margin-right: 10px; transform: scale(1.2); }
            .phajay-option span { font-size: 15px; font-weight: 500; color: #333; }
            
            .phajay-bank-select {
                margin-left: 28px; margin-bottom: 15px; display: none;
                padding: 8px; width: 80%; border-radius: 4px; border: 1px solid #ccc; font-size: 14px;
            }

            .phajay-btn {
                background-color: <?php echo esc_attr($bg); ?> !important;
                color: <?php echo esc_attr($text); ?> !important;
                border-radius: <?php echo esc_attr($radius); ?>px !important;
                padding: 12px 24px; border: none; cursor: pointer; text-decoration: none;
                font-size: 16px; width: 100%; text-align: center; margin-top: 10px; font-weight: bold;
            }
            .phajay-btn:hover { opacity: 0.9; }
        </style>
        <script>
            function toggleBankSelect(val) {
                var bankDiv = document.getElementById('phajay-bank-dropdown');
                if(bankDiv) {
                    if(val === 'qr') {
                        bankDiv.style.display = 'block';
                    } else {
                        bankDiv.style.display = 'none';
                    }
                }
            }
        </script>
        <?php
    }

    // --- 3. Shortcode ---
    public function render_shortcode($atts) {
        $default_label = get_option('phajay_btn_label', 'ดำเนินการชำระเงิน');
        
        $a = shortcode_atts(array(
            'amount' => '100',
            'description' => 'Payment',
            'label' => $default_label
        ), $atts);

        // รับค่า Dynamic Amount
        if (isset($_GET['pay_amount']) && is_numeric($_GET['pay_amount'])) {
            $a['amount'] = floatval($_GET['pay_amount']);
        }

        $show_link = get_option('phajay_enable_link', '1');
        $show_qr = get_option('phajay_enable_qr', '1');
        $show_credit = get_option('phajay_enable_credit', '1');

        ob_start();
        ?>
        <form action="<?php echo esc_url(admin_url('admin-post.php')); ?>" method="post" class="phajay-wrapper">
            <input type="hidden" name="action" value="phajay_process_payment">
            <input type="hidden" name="amount" value="<?php echo esc_attr($a['amount']); ?>">
            <input type="hidden" name="description" value="<?php echo esc_attr($a['description']); ?>">
            
            <div style="margin-bottom:15px;padding:10px;background:#fff;border-radius:5px;border:1px solid #ddd;text-align:center;">
                ยอดชำระ: <strong style="color:#28a745;font-size:18px;"><?php echo number_format($a['amount']); ?> LAK</strong>
            </div>

            <p style="margin-top:0;margin-bottom:15px;font-weight:bold;color:#555;">เลือกวิธีชำระเงิน:</p>

            <?php if ($show_link) : ?>
                <label class="phajay-option">
                    <input type="radio" name="payment_method" value="link" checked onclick="toggleBankSelect('link')">
                    <span>Payment Link (หน้าเว็บรวม)</span>
                </label>
            <?php endif; ?>

            <?php if ($show_qr) : ?>
                <label class="phajay-option">
                    <input type="radio" name="payment_method" value="qr" onclick="toggleBankSelect('qr')">
                    <span>Scan QR (สแกนจ่ายทันที)</span>
                </label>
                <select name="selected_bank" id="phajay-bank-dropdown" class="phajay-bank-select">
                    <option value="bcel">BCEL OnePay</option>
                    <option value="jdb">JDB Bank</option>
                    <option value="ldb">LDB Bank</option>
                    <option value="ib">Indochina Bank</option>
                </select>
            <?php endif; ?>

            <?php if ($show_credit) : ?>
                <label class="phajay-option">
                    <input type="radio" name="payment_method" value="credit" onclick="toggleBankSelect('credit')">
                    <span>Credit Card (บัตรเครดิต)</span>
                </label>
            <?php endif; ?>

            <button type="submit" class="phajay-btn"><?php echo esc_html($a['label']); ?></button>
        </form>
        <?php
        return ob_get_clean();
    }

    // --- 4. Process Payment ---
    public function process_payment() {
        $secret_key = get_option('phajay_secret_key');
        if (empty($secret_key)) wp_die('Error: Missing Secret Key');

        $amount = floatval($_POST['amount']);
        $desc = sanitize_text_field($_POST['description']);
        $method = sanitize_text_field($_POST['payment_method']);
        $order_no = 'ORD-' . time() . rand(100,999);

        // A. Link
        if ($method === 'link') {
            $url = $this->api_link_url;
            $body = array('orderNo' => $order_no, 'amount' => $amount, 'description' => $desc);
            $this->call_api_and_redirect($url, $body, $secret_key, true, 'redirectURL');
        } 
        // B. Credit Card
        elseif ($method === 'credit') {
            $url = $this->api_credit_url;
            $body = array('amount' => $amount, 'description' => $desc);
            $this->call_api_and_redirect($url, $body, $secret_key, true, 'paymentUrl');
        }
        // C. QR Code
        elseif ($method === 'qr') {
            $bank = isset($_POST['selected_bank']) ? sanitize_text_field($_POST['selected_bank']) : 'bcel';
            if (!array_key_exists($bank, $this->api_qr_endpoints)) wp_die("Error: Unsupported bank");
            
            $url = $this->api_qr_endpoints[$bank];
            $response = wp_remote_post($url, array(
                'headers' => array('Content-Type' => 'application/json', 'secretKey' => $secret_key),
                'body' => json_encode(array('amount' => $amount, 'description' => $desc)),
                'timeout' => 45
            ));

            if (is_wp_error($response)) wp_die($response->get_error_message());
            $result = json_decode(wp_remote_retrieve_body($response), true);

            if (!empty($result['qrCode'])) {
                $this->render_qr_page($result['qrCode'], $bank, $amount, $result);
            } else {
                wp_die('Error generating QR: ' . print_r($result, true));
            }
        }
    }

    private function call_api_and_redirect($url, $body, $secret_key, $is_basic_auth, $redirect_key) {
        $headers = array('Content-Type' => 'application/json');
        if ($is_basic_auth) {
            $headers['Authorization'] = 'Basic ' . base64_encode($secret_key);
        } else {
            $headers['secretKey'] = $secret_key;
        }

        $response = wp_remote_post($url, array(
            'headers' => $headers,
            'body' => json_encode($body),
            'timeout' => 45
        ));

        if (is_wp_error($response)) wp_die($response->get_error_message());
        $result = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($result[$redirect_key])) {
            wp_redirect($result[$redirect_key]);
            exit;
        } else {
            wp_die('API Error: ' . print_r($result, true));
        }
    }

    private function render_qr_page($qr_string, $bank, $amount, $full_result) {
        $qr_image_url = 'https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=' . urlencode($qr_string);
        $deep_link = isset($full_result['link']) ? $full_result['link'] : '#';
        $bank_upper = strtoupper($bank);

        echo '<!DOCTYPE html><html><head><meta name="viewport" content="width=device-width, initial-scale=1"><title>Scan to Pay</title>';
        echo '<style>body{font-family:sans-serif;background:#f0f2f5;display:flex;justify-content:center;align-items:center;height:100vh;margin:0;} 
              .box{background:white;padding:30px;border-radius:15px;text-align:center;box-shadow:0 5px 15px rgba(0,0,0,0.1);width:320px;}
              .btn{display:block;width:100%;padding:12px;background:#0056b3;color:white;text-decoration:none;border-radius:8px;margin-top:15px;font-weight:bold;}
              </style></head><body>';
        echo '<div class="box">';
        echo "<h3>สแกนจ่าย $bank_upper</h3>";
        echo '<img src="' . $qr_image_url . '" style="width:100%;border-radius:8px;">';
        echo '<p>ยอดชำระ: <strong>' . number_format($amount) . ' LAK</strong></p>';
        if($deep_link != '#' && $bank != 'ib') {
            echo '<a href="' . $deep_link . '" class="btn">เปิดแอปธนาคาร</a>';
        }
        echo '<br><a href="' . home_url() . '" style="color:#888;text-decoration:none;font-size:13px;">&larr; กลับ</a>';
        echo '</div></body></html>';
        exit;
    }
}

new Phajay_Payment_Plugin();